<?php

defined('SYSPATH') or die('No direct script access.');

abstract class Widget_Form {
    /**
     * @var Widget_Form_Field $field
     */
    /**
     * @var string DEFAULT_VIEW default View template
     */
    const DEFAULT_VIEW = 'common/form';
    const DEFAULT_ERRORS = 'validate';

    /**
     * @property array $fields array of Widget_Form_Field objects; definitions of
     * the fields in the form
     */
    protected $fields = array();
    /**
     * @property array $buttons array of Widget_Form_Button objects; buttons
     * which (with default view) will be shown ubder the rest of the form
     */
    protected $buttons = array();
    /**
     * @var Validate $validate validate object
     * @var bool $validated flag inditacting wether object was already validated
     */

    /**
     * @property names of fields that will be excluded from get_validated_array();
     * useful for hiding fields from orms
     */
    protected $exclude_fields = array();

    /** @property Validate $validate Kohana Validation object*/
    protected $validate = NULL;
    protected $validated = FALSE;
    protected $valid = NULL;
    protected $view = NULL;
    protected $action = '';
    protected $attributes = array();
    protected $has_files = NULL;

    /**
     * @return Validate validate object after check
     */
    public function get_validated() {
        $this->validate();
        return $this->validate;
    }

    public function get_validated_array() {
        $array = $this->get_validated()->as_array();
        foreach($this->exclude_fields as $field){
            unset($array[$field]);
        }
        return $array;
    }

    public function get_fields() {
        $this->build_form();
        return $this->fields;
    }

    public function get_buttons() {
        $this->build_form();
        return $this->buttons;
    }

    public function get_errors($file = NULL) {
        if (empty($file)) {
            $file = self::DEFAULT_ERRORS;
        }
        $this->validate();
        return $this->validate->errors($file);
    }

    public function add_error($field, $error, array $params = NULL){
        $this->validate->error($field, $error, $params);
        $this->fields[$field]->set_error($error);
    }

    /**
     * set form values
     * @param array $values to fill the form fields
     */
//protected function set_values(array $values = array(), array $files = array());

    /**
     * sets view to be used for form rendering
     * @param mixed $view View to be used (string or an instance of Kohana_View)
     */
    public function set_view($view) {
        if ($view instanceof Kohana_View) {
            $this->view = $view;
        } elseif (is_string($view)) {
            $this->view = View::factory($view);
        } else {
            throw new Exception('Unsupported type of $view given.');
        }
    }

    public function __toString() {
        return (string) $this->render();
    }

    /**
     * render form
     * @return string rendered View
     */
    public function render() {
        if ($this->view === NULL) {
            $this->set_view(self::DEFAULT_VIEW);
        }
        $this->build_form();

        $this->view->fields = $this->fields;
        $this->view->form = $this;
        $this->view->buttons = $this->buttons;

        return $this->view;
    }

    /**
     * @return bool true if object is valid
     */
    public function is_valid() {
        return $this->validate();
    }

    public function __construct($action = '', $values = array(), array $attributes = array(), $files = array()) {
        $this->attributes = $attributes;
        $this->validate = Validate::factory(array_merge($values, $files));
        $this->action = $action;
    }

    /**
     * Function to check wether form is already built
     * @return bool true if form is built; false otherwise
     */
    protected function is_built() {
        return!empty($this->fields);
    }

    protected function build_form($force = FALSE) {
        if ($force OR $this->is_built())
            return;
        $this->prepare_fields();
        $this->prepare_buttons();
        $this->prepare_labels();
        $this->sync_values();
        $this->apply_rules_from_fields();
        $this->apply_callbacks_from_fields();
        if ($this->has_files()) {
            $this->attributes['enctype'] = 'multipart/form-data';
        }
    }

    /**
     * for all fields; if value for the field is in the validate object update
     * the value in the field; otherwise get vaule from the field as default
     */
    protected function sync_values() {
        foreach ($this->fields as $field) {
            if ($this->validate->offsetExists($field->name)) {
                $field->value = $this->validate[$field->name];
            } else {
                $this->validate[$field->name] = $field->value;
            }
        }
    }
    
    

    protected function set_field_errors() {
//TODO make form error definitions
        $errors = $this->get_errors();
        foreach ($this->fields as $field) {
            if (array_key_exists($field->name, $errors)) {
                $field->set_error($errors[$field->name]);
            }
        }
    }

    /**
     * iterate thorugh fields and apply rules saved there
     */
    protected function apply_rules_from_fields() {
        foreach ($this->fields as $field) {
            foreach ($field->get_rules() as $fieldname => $fieldrules) {
                foreach ($fieldrules as $rule => $params) {
                    $this->validate->rule($fieldname, $rule, $params);
                }
            }
        }
    }

    /**
     * TODO: completetly untested; never used yet
     * iterate thorugh fields and apply callbacks saved there
     */
    protected function apply_callbacks_from_fields() {
        foreach ($this->fields as $field) {
            foreach ($field->get_callbacks() as $callback) {
                $this->validate->callback($field->name, $callback);
            }
        }
    }

    /**
     * validate the form if necessary; set the validated flag and return boolean
     * result of validation
     * @return bool true if object is valid
     */
    protected function validate() {
        if (!$this->validated) {
            $this->build_form();
            $this->valid = ($this->validate->check() && $this->custom_validation());
            $this->validated = TRUE;
            $this->set_field_errors();
        }
        return $this->valid;
    }

    public function open() {
        return Form::open($this->action, $this->attributes);
    }

    public function close() {
        return Form::close();
    }

    public function has_files() {
        if (!is_null($this->has_files))
            return $this->has_files;

        foreach ($this->fields as $field) {
            if($field instanceof Widget_Form_Field_File)
                return TRUE;
        }
    }

    protected function custom_validation() {
        return TRUE;
    }

    protected function prepare_labels() {
        foreach ($this->fields as $field) {
            foreach ($field->get_labels() as $fieldname => $label) {
                $this->validate->label($fieldname, $label->get_text());
            }
        }
    }

    abstract protected function prepare_fields();

    abstract protected function prepare_buttons();

    public static function is_field_unique($value, $field_name, $model, $id = NULL) {
        if (ORM::factory($model)->is_unique($field_name, $value, $id)) {
            return FALSE;
        }
        return TRUE;
    }

    public static function is_field_selected($value, $default = 0) {
        if ($value == $default) {
            return FALSE;
        }
        return TRUE;
    }

}

?>
