<?php

defined('SYSPATH') or die('No direct script access.');

/**
 * @author luk
 */
class Model_Repository extends ORM {
    const BASE_FILE_DIR='media/upload/repository/';
    const MODE = 0777;

    protected $_has_many = array('files' => array('through' => 'repository_file'));
    protected $_belongs_to = array('repository_kind' => array());

    public static function get_repo_id_from_kind_and_element($kind_id, $element_id) {
        $repo = ORM::factory('Repository')->where('repository_kind_id', '=', $kind_id)
                        ->where('element_id', '=', $element_id)
                        ->find();
        return $repo->id;
    }

    /**
     * Constructor
     * @param array $id array of group_kind_id and element_id to guess the apropriate repository e.g. array( 'kind_id' => $kind_id, 'element_id' => $site_id )
     * @param int $id id of the repository
     */
    public function __construct($id = NULL) {
        if (is_array($id)) {
            $id = Model_Repository::get_repo_id_from_kind_and_element($id['kind_id'], $id['element_id']);
        }
        parent::__construct($id);
    }

    public function save() {
        $new = $this->is_new();
        parent::save();
        if ($new) {
            mkdir($this->get_path(), self::MODE, TRUE);
        }
        return $this;
    }

    /**
     * @return string get path to directory of this repository
     */
    public function get_path() {
        return Model_Repository::BASE_FILE_DIR . $this->repository_kind_id . '/' . $this->element_id . '/';
    }

    public function get_owner_model() {
        $modelname = $this->repository_kind->name;
        $model = ORM::factory($modelname, $this->element_id);
        return $model;
    }

    /**
     * save given file in the repository
     * @param <type> $file file being added, should be compatible with $_FILE['']
     * @param array $fields optional array of other fields(colums) to set; interpreted as array( [ $coulumn => $value, ... ] )
     */
    public function save_file($file, array $fields = array()) {
        $fname = $this->prepare_file_name($file['name']);
        $fpath = $this->get_path();
        $file_orm = ORM::factory('File');
        $fname = $this->prepare_name($fname);
        $file_orm->name = $fname;
        $file_orm->path = $fpath;
        $file_orm->repository = $this;
        foreach ($fields as $field => $value) {
            $file_orm->$field = $value;
        }
        Upload::save($file, $fname, $fpath);
        return $file_orm->save();
    }

    private function prepare_file_name($original_name) {
        return preg_replace('/\s+/', '_', UTF8::transliterate_to_ascii($original_name));
    }

    public function delete($id = NULL) {

        /* a preliminary get for path must be performed
         * cause the data to determine the repository
         * path might be deleted before retrieving */
        $path = $this->get_path();

        if ($id === NULL) {
            $id = $this->pk();
        }
        if ($this->delete_self($id) == 0) {
            return $this;
        }
        if ($this->is_new()) {
            throw new Exception('Trying to delete a not yet created repository.');
        }
        $this->delete_relations($id);
        self::rrmdir($path);
        return $this;
    }

    /**
     * returns element associated with the repository
     * @return Model elemenet associated with the repository 
     */
    public function get_element() {
        $model_name = $this->repository_kind->name;
        $model = ORM::factory($model_name, $this->element_id);
        return $model;
    }

    /**
     * removes appropirate relations of the repository being deleted
     * $param mixed $id id of the repository being destroyed
     * @var Model_File $file file being removed from repository
     */
    protected function delete_relations($id) {
        $files_in_repo = $this->files->find_all();
        foreach ($files_in_repo as $file) {
            $file->remove_from_repository($this, TRUE);
        }
    }

    protected function prepare_name($fname) {
        if ($this->file_exists($fname)) {
            $dot_pos = strrpos($fname, '.');
            if ($dot_pos)
                $name = str_split($fname, $dot_pos);
            else {
                $name[0] = $fname;
                $name[1] = '';
            }
            $iter = 1;
            do {
                $iter++;
                $fname = $name[0] . '(' . $iter . ')' . $name[1];
            } while ($this->file_exists($fname));
        }
        return $fname;
    }

    /**
     * checks if the file with given name already exists within the repository
     * @param string $fname filename to check
     * @return number of files in the repository with given name should be 0 or 1
     */
    protected function file_exists($fname) {
        return $this->files->where('name', '=', $fname)->count_all();
    }

    /**
     * Helper function for recursive deletion of a directory
     * @param <type> $dir directory
     */
    protected static function rrmdir($dir) {
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != "." && $object != "..") {
                    if (filetype($dir . "/" . $object) == "dir") {
                        self::rrmdir($dir . "/" . $object);
                    } else {
                        unlink($dir . "/" . $object);
                    }
                }
            }
            reset($objects);
            rmdir($dir);
        }
    }

}

?>
